﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;

namespace YumeNikkiRandomizer
{
    class ChipSets : RPGByteData
    {
        List<ChipSet> chipSets;
        
        public ChipSets(FileStream f)
        {
            load(f);
        }
        public ChipSets()
        {
        }
        
        override public void load(FileStream f)
        {
            chipSets = M.readDatabaseList<ChipSet>(f, "ChipSets", "ChipSet", ref M.chipSetNames);
        }
        
        override protected void myWrite()
        {
            M.writeListNoLength<ChipSet>(chipSets);
        }
    }
    
    class ChipSet : RPGDatabaseEntry
    {
        int id = 0;
        string chipSetName = ""; // 01
        string chipSetFile = ""; // 02
        int[] lowerTerrain; // 03
        List<ChipSetChip> lowerChips; // 04
        List<ChipSetChip> upperChips; // 05
        int seaSequence = 0; // 0b
        int seaSpeed = 0; // 0c
        
        static string myClass = "ChipSet";
        Chunks chunks;
        
        public ChipSet(FileStream f)
        {
            load(f);
        }
        public ChipSet()
        {
        }
        
        public void load(FileStream f)
        {
            chunks = new Chunks(f, myClass);
            
            id = M.readMultibyte(f);
            
            if (chunks.next(0x01))
                chipSetName = M.readStringDataName(f, id, ref M.chipSetNames, M.S_TOTRANSLATE);
            
            if (chunks.next(0x02))
                chipSetFile = M.readString(f, M.S_FILENAME);
            
            if (chunks.next(0x03))
                lowerTerrain = M.readTwoByteArray(f);
            if (chunks.next(0x04))
                lowerChips = M.readListNoLength<ChipSetChip>(f);
            if (chunks.next(0x05))
                upperChips = M.readListNoLength<ChipSetChip>(f);
            
            if (chunks.next(0x0b))
                seaSequence = M.readLengthMultibyte(f);
            if (chunks.next(0x0c))
                seaSpeed = M.readLengthMultibyte(f);
            
            M.byteCheck(f, 0x00);
        }
        
        public void write()
        {
            M.writeMultibyte(id);
            
            if (chunks.wasNext(0x01))
                M.writeString(chipSetName, M.S_TOTRANSLATE);
            
            if (chunks.wasNext(0x02))
                M.writeString(chipSetFile, M.S_FILENAME);
            
            if (chunks.wasNext(0x03))
                M.writeTwoByteArray(lowerTerrain);
            if (chunks.wasNext(0x04))
                M.writeListNoLength<ChipSetChip>(lowerChips);
            if (chunks.wasNext(0x05))
                M.writeListNoLength<ChipSetChip>(upperChips);
            
            if (chunks.wasNext(0x0b))
                M.writeLengthMultibyte(seaSequence);
            if (chunks.wasNext(0x0c))
                M.writeLengthMultibyte(seaSpeed);
            
            M.writeByte(0x00);
        }
        
        public bool isBlank()
        {
            if (chipSetName != "" // 01
             || chipSetFile != "" // 02
             // 03 below
             // 04 below
             // 05 below
             || seaSequence != 0 // 0b
             || seaSpeed != 0) // 0c
                return false;
            
            if (lowerTerrain != null) // 03
                for (int i = 0; i < lowerTerrain.Length; i++)
                    if (lowerTerrain[i] != 1)
                        return false;
            
            if (lowerChips != null) // 04
                foreach (ChipSetChip chip in lowerChips)
                    if (!chip.isBlank())
                        return false;
            
            if (upperChips != null) // 05
                foreach (ChipSetChip chip in upperChips)
                    if (!chip.isBlank())
                        return false;
            
            return true;
        }
    }
    
    class ChipSetChip : RPGData
    {
        bool down = false;
        bool left = false;
        bool right = false;
        bool up = false;
        bool star = false;
        bool square = false;
        bool counter = false;
        int passType = 0;
        
        public ChipSetChip(FileStream f)
        {
            load(f);
        }
        public ChipSetChip()
        {
        }
        
        public void load(FileStream f)
        {
            int c = M.readByte(f);
            down = (c & 1) != 0;
            left = (c & 2) != 0;
            right = (c & 4) != 0;
            up = (c & 8) != 0;
            star = (c & 16) != 0;
            square = (c & 32) != 0;
            counter = (c & 64) != 0;
            
            passType = 0; // X
            if (square)
                passType = 1; // Square
            else if (star)
                passType = 2; // Star
            else if (down || left || right || up)
                passType = 3; // Circle
        }
        
        public void write()
        {
            int c = (down? 1 : 0)
                  + (left? 2 : 0)
                  + (right? 4 : 0)
                  + (up? 8 : 0)
                  + (star? 16 : 0)
                  + (square? 32 : 0)
                  + (counter? 64 : 0);
            
            M.writeByte(c);
        }
        
        public bool isBlank()
        {
            if (down
             || left
             || right
             || up
             || star
             || square
             || counter
             || passType != 0)
                return false;
            
            return true;
        }
    }
}
